/*
 * SPDX-License-Identifier: MIT
 * Copyright (C) 2004 - 2021 AJA Video Systems, Inc.
 */
//==========================================================================
//
//  ntv2hdmiedid.c
//
//==========================================================================

#include "ntv2hdmiedid.h"

/* debug messages */
#define NTV2_DEBUG_INFO					0x00000001
#define NTV2_DEBUG_ERROR				0x00000002

#define NTV2_DEBUG_ACTIVE(msg_mask) \
	((ntv2_debug_mask & msg_mask) != 0)

#define NTV2_MSG_PRINT(msg_mask, string, ...) \
	if(NTV2_DEBUG_ACTIVE(msg_mask)) ntv2Message(string, __VA_ARGS__);

#define NTV2_MSG_INFO(string, ...)					NTV2_MSG_PRINT(NTV2_DEBUG_INFO, string, __VA_ARGS__)
#define NTV2_MSG_ERROR(string, ...)					NTV2_MSG_PRINT(NTV2_DEBUG_ERROR, string, __VA_ARGS__)

static uint32_t ntv2_debug_mask = 0;

static uint8_t c_hdmi_port_pefix[] = { 0x03, 0x0c, 0x00 };
static uint8_t c_hdmi_port_number[] = { 0x10, 0x20, 0x30, 0x40 };

static uint8_t c_edid_konahdmi_20[] = {
	0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x05, 0x41, 0x2E, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x0F, 0x1C, 0x01, 0x03, 0x80, 0xA0, 0x5A, 0x78, 0x0A, 0xEE, 0x91, 0xA3, 0x54, 0x4C, 0x99, 0x26,
	0x0F, 0x50, 0x54, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x08, 0xE8, 0x00, 0x30, 0xF2, 0x70, 0x5A, 0x80, 0xB0, 0x58,
	0x8A, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x02, 0x3A, 0x80, 0x18, 0x71, 0x38, 0x2D, 0x40,
	0x58, 0x2C, 0x45, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0xFD, 0x00, 0x17,
	0x3D, 0x0F, 0x88, 0x3C, 0x00, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFC,
	0x00, 0x4B, 0x6F, 0x6E, 0x61, 0x20, 0x48, 0x44, 0x4D, 0x49, 0x0A, 0x20, 0x20, 0x20, 0x01, 0xFC,
	
	0x02, 0x03, 0x5D, 0x51, 0x57, 0x61, 0x60, 0x5F, 0x5E, 0x5D, 0x66, 0x65, 0x64, 0x63, 0x62, 0x10,
	0x1F, 0x22, 0x21, 0x20, 0x05, 0x14, 0x04, 0x13, 0x06, 0x15, 0x07, 0x16, 0x23, 0x0F, 0x07, 0x07,
	0x83, 0x4F, 0x00, 0x00, 0x70, 0x03, 0x0C, 0x00, 0x10, 0x00, 0x30, 0x3C, 0x20, 0xA0, 0x82, 0x01,
	0x02, 0x03, 0x04, 0x01, 0x41, 0x67, 0xD8, 0x5D, 0xC4, 0x01, 0x78, 0x80, 0x03, 0xE2, 0x00, 0x73,
	0xE3, 0x05, 0xE0, 0x00, 0xE5, 0x0F, 0x63, 0x00, 0x00, 0x00, 0xE3, 0x06, 0x3F, 0x01, 0xEE, 0x01,
	0x46, 0xD0, 0x00, 0x24, 0x0F, 0x8B, 0x00, 0xA8, 0x53, 0x4B, 0x9D, 0x27, 0x0B, 0x01, 0x1D, 0x00,
	0xBE, 0x82, 0x38, 0x2D, 0x40, 0x7E, 0x2C, 0x45, 0x80, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1E
};

static uint8_t c_edid_konahdmi_13[] = {
	0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x05, 0x41, 0x2E, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x0F, 0x1C, 0x01, 0x03, 0x80, 0xA0, 0x5A, 0x78, 0x0A, 0xEE, 0x91, 0xA3, 0x54, 0x4C, 0x99, 0x26,
	0x0F, 0x50, 0x54, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3A, 0x80, 0x18, 0x71, 0x38, 0x2D, 0x40, 0x58, 0x2C,
	0x45, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x30, 0x30, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0xFD, 0x00, 0x17,
	0x3D, 0x0F, 0x44, 0x1E, 0x00, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFC,
	0x00, 0x4B, 0x6F, 0x6E, 0x61, 0x20, 0x48, 0x44, 0x4D, 0x49, 0x0A, 0x20, 0x20, 0x20, 0x01, 0x0E,
	
	0x02, 0x03, 0x3C, 0x51, 0x4D, 0x10, 0x1F, 0x22, 0x21, 0x20, 0x05, 0x14, 0x04, 0x13, 0x06, 0x15,
	0x07, 0x16, 0x23, 0x0F, 0x07, 0x07, 0x83, 0x4F, 0x00, 0x00, 0x67, 0x03, 0x0C, 0x00, 0x10, 0x00,
	0x30, 0x3C, 0xE2, 0x00, 0x73, 0xE3, 0x05, 0xE0, 0x00, 0xE3, 0x06, 0x3F, 0x01, 0xEE, 0x01, 0x46,
	0xD0, 0x00, 0x24, 0x0F, 0x8B, 0x00, 0xA8, 0x53, 0x4B, 0x9D, 0x27, 0x0B, 0x01, 0x1D, 0x00, 0xBE,
	0x82, 0x38, 0x2D, 0x40, 0x7E, 0x2C, 0x45, 0x80, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x62
};

static uint8_t c_edid_corvidhbr[] = {
	0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x05, 0x41, 0x18, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x2E, 0x19, 0x01, 0x03, 0x80, 0xA0, 0x5A, 0x78, 0x0A, 0xEE, 0x91, 0xA3, 0x54, 0x4C, 0x99, 0x26,
	0x0F, 0x50, 0x54, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x1D, 0x80, 0x3E, 0x73, 0x38, 0x2D, 0x40, 0x7E, 0x2C,
	0x45, 0x80, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x02, 0x3A, 0x80, 0x18, 0x71, 0x38, 0x2D, 0x40,
	0x58, 0x2C, 0x45, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0xFD, 0x00, 0x17,
	0x3D, 0x0F, 0x88, 0x1E, 0x00, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFC,
	0x00, 0x43, 0x6F, 0x72, 0x76, 0x69, 0x64, 0x20, 0x48, 0x42, 0x2D, 0x52, 0x0A, 0x20, 0x01, 0x1A,
	
	0x02, 0x03, 0x44, 0x71, 0x55, 0x61, 0x60, 0x5F, 0x5E, 0x5D, 0x66, 0x65, 0x64, 0x63, 0x62, 0x10,
	0x1F, 0x22, 0x21, 0x20, 0x05, 0x14, 0x04, 0x13, 0x06, 0x15, 0x23, 0x0F, 0x04, 0x07, 0x83, 0x4F,
	0x00, 0x00, 0x70, 0x03, 0x0C, 0x00, 0x10, 0x00, 0x10, 0x3C, 0x20, 0xA0, 0x82, 0x01, 0x02, 0x03,
	0x04, 0x01, 0x41, 0xE2, 0x00, 0x73, 0xE3, 0x05, 0xE0, 0x00, 0xE5, 0x0F, 0x63, 0x00, 0x00, 0x00,
	0xE3, 0x06, 0x0F, 0x01, 0x01, 0x1D, 0x00, 0xBE, 0x82, 0x38, 0x2D, 0x40, 0x7E, 0x2C, 0x45, 0x80,
	0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBD
};

static uint8_t c_edid_io4k[] = {
	0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x05, 0x41, 0x18, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x2E, 0x19, 0x01, 0x03, 0x80, 0xA0, 0x5A, 0x78, 0x0A, 0xEE, 0x91, 0xA3, 0x54, 0x4C, 0x99, 0x26,
	0x0F, 0x50, 0x54, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x1D, 0x80, 0x3E, 0x73, 0x38, 0x2D, 0x40, 0x7E, 0x2C,
	0x45, 0x80, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x02, 0x3A, 0x80, 0x18, 0x71, 0x38, 0x2D, 0x40,
	0x58, 0x2C, 0x45, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0xFD, 0x00, 0x17,
	0x3D, 0x0F, 0x88, 0x1E, 0x00, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFC,
	0x00, 0x49, 0x6F, 0x20, 0x34, 0x4B, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x01, 0x93,
	
	0x02, 0x03, 0x44, 0x71, 0x55, 0x61, 0x60, 0x5F, 0x5E, 0x5D, 0x66, 0x65, 0x64, 0x63, 0x62, 0x10,
	0x1F, 0x22, 0x21, 0x20, 0x05, 0x14, 0x04, 0x13, 0x06, 0x15, 0x23, 0x0F, 0x04, 0x07, 0x83, 0x4F,
	0x00, 0x00, 0x70, 0x03, 0x0C, 0x00, 0x10, 0x00, 0x10, 0x3C, 0x20, 0xA0, 0x82, 0x01, 0x02, 0x03,
	0x04, 0x01, 0x41, 0xE2, 0x00, 0x73, 0xE3, 0x05, 0xE0, 0x00, 0xE5, 0x0F, 0x63, 0x00, 0x00, 0x00,
	0xE3, 0x06, 0x0F, 0x01, 0x01, 0x1D, 0x00, 0xBE, 0x82, 0x38, 0x2D, 0x40, 0x7E, 0x2C, 0x45, 0x80,
	0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xBD
};

static uint8_t c_edid_io4kplus[] = {
	0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x05, 0x41, 0x2B, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x0F, 0x1C, 0x01, 0x03, 0x80, 0xA0, 0x5A, 0x78, 0x0A, 0xEE, 0x91, 0xA3, 0x54, 0x4C, 0x99, 0x26,
	0x0F, 0x50, 0x54, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x08, 0xE8, 0x00, 0x30, 0xF2, 0x70, 0x5A, 0x80, 0xB0, 0x58,
	0x8A, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x02, 0x3A, 0x80, 0x18, 0x71, 0x38, 0x2D, 0x40,
	0x58, 0x2C, 0x45, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0xFD, 0x00, 0x17,
	0x3D, 0x0F, 0x88, 0x3C, 0x00, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFC,
	0x00, 0x49, 0x4F, 0x20, 0x34, 0x4B, 0x20, 0x50, 0x6C, 0x75, 0x73, 0x0A, 0x20, 0x20, 0x01, 0xEF,

	0x02, 0x03, 0x5D, 0x51, 0x57, 0x61, 0x60, 0x5F, 0x5E, 0x5D, 0x66, 0x65, 0x64, 0x63, 0x62, 0x10,
	0x1F, 0x22, 0x21, 0x20, 0x05, 0x14, 0x04, 0x13, 0x06, 0x15, 0x07, 0x16, 0x23, 0x0F, 0x07, 0x07,
	0x83, 0x4F, 0x00, 0x00, 0x70, 0x03, 0x0C, 0x00, 0x10, 0x00, 0x30, 0x3C, 0x20, 0xA0, 0x82, 0x01,
	0x02, 0x03, 0x04, 0x01, 0x41, 0x67, 0xD8, 0x5D, 0xC4, 0x01, 0x78, 0x80, 0x03, 0xE2, 0x00, 0x73,
	0xE3, 0x05, 0xE0, 0x00, 0xE5, 0x0F, 0x63, 0x00, 0x00, 0x00, 0xE3, 0x06, 0x3F, 0x01, 0xEE, 0x01,
	0x46, 0xD0, 0x00, 0x24, 0x0F, 0x8B, 0x00, 0xA8, 0x53, 0x4B, 0x9D, 0x27, 0x0B, 0x01, 0x1D, 0x00,
	0xBE, 0x82, 0x38, 0x2D, 0x40, 0x7E, 0x2C, 0x45, 0x80, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1E
};

static uint8_t c_edid_iox3[] = {
	0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x05, 0x41, 0x2E, 0x00, 0x01, 0x01, 0x01, 0x01,
	0x0F, 0x1C, 0x01, 0x03, 0x80, 0xA0, 0x5A, 0x78, 0x0A, 0xEE, 0x91, 0xA3, 0x54, 0x4C, 0x99, 0x26,
	0x0F, 0x50, 0x54, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3A, 0x80, 0x18, 0x71, 0x38, 0x2D, 0x40, 0x58, 0x2C,
	0x45, 0x00, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x30, 0x30, 0x30,
	0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0xFD, 0x00, 0x17,
	0x3D, 0x0F, 0x44, 0x1E, 0x00, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xFC,
	0x00, 0x49, 0x4F, 0x58, 0x33, 0x0A, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x01, 0x16,

	0x02, 0x03, 0x3C, 0x51, 0x4D, 0x10, 0x1F, 0x22, 0x21, 0x20, 0x05, 0x14, 0x04, 0x13, 0x06, 0x15,
	0x07, 0x16, 0x23, 0x0F, 0x07, 0x07, 0x83, 0x4F, 0x00, 0x00, 0x67, 0x03, 0x0C, 0x00, 0x10, 0x00,
	0x30, 0x3C, 0xE2, 0x00, 0x73, 0xE3, 0x05, 0xE0, 0x00, 0xE3, 0x06, 0x3F, 0x01, 0xEE, 0x01, 0x46,
	0xD0, 0x00, 0x24, 0x0F, 0x8B, 0x00, 0xA8, 0x53, 0x4B, 0x9D, 0x27, 0x0B, 0x01, 0x1D, 0x00, 0xBE,
	0x82, 0x38, 0x2D, 0x40, 0x7E, 0x2C, 0x45, 0x80, 0x20, 0xC2, 0x31, 0x00, 0x00, 0x1E, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x62
};


struct ntv2_hdmiedid *ntv2_hdmiedid_open(Ntv2SystemContext* sys_con,
										 const char *name, int index)
{
	struct ntv2_hdmiedid *ntv2_hed = NULL;

	if ((sys_con == NULL) ||
		(name == NULL))
		return NULL;

	ntv2_hed = (struct ntv2_hdmiedid *)ntv2MemoryAlloc(sizeof(struct ntv2_hdmiedid));
	if (ntv2_hed == NULL) {
		NTV2_MSG_ERROR("%s: ntv2hdmiedid instance memory allocation failed\n", name);
		return NULL;
	}
	memset(ntv2_hed, 0, sizeof(struct ntv2_hdmiedid));

	ntv2_hed->index = index;
#if defined(MSWindows)
	sprintf(ntv2_hed->name, "%s%d", name, index);
#else
	snprintf(ntv2_hed->name, NTV2_HDMIEDID_STRING_SIZE, "%s%d", name, index);
#endif
	ntv2_hed->system_context = sys_con;

	NTV2_MSG_INFO("%s: open ntv2_hdmiedid\n", ntv2_hed->name);

	return ntv2_hed;
}

void ntv2_hdmiedid_close(struct ntv2_hdmiedid *ntv2_hed)
{
	if (ntv2_hed == NULL) 
		return;

	NTV2_MSG_INFO("%s: close ntv2_hdmiedid\n", ntv2_hed->name);

	memset(ntv2_hed, 0, sizeof(struct ntv2_hdmiedid));
	ntv2MemoryFree(ntv2_hed, sizeof(struct ntv2_hdmiedid));
}

Ntv2Status ntv2_hdmiedid_configure(struct ntv2_hdmiedid *ntv2_hed,
								   enum ntv2_edid_type type,
								   int port_index)
{
	uint8_t *edid;
	uint32_t size;
	uint32_t start;
	uint32_t end;
	uint32_t count;
	uint32_t prefix_size;
	uint32_t edid_size;
	uint32_t sum;
	bool modify_port = false;
	uint32_t i;
	uint32_t j;
	
	if (ntv2_hed == NULL) 
		return NTV2_STATUS_FAIL;

	NTV2_MSG_INFO("%s: configure hdmiedid\n", ntv2_hed->name);

	switch (type)
	{
	case ntv2_edid_type_konahdmi_20:
		edid = c_edid_konahdmi_20;
		size = sizeof(c_edid_konahdmi_20);
		modify_port = true;
		break;
	case ntv2_edid_type_konahdmi_13:
		edid = c_edid_konahdmi_13;
		size = sizeof(c_edid_konahdmi_13);
		modify_port = true;
		break;
	case ntv2_edid_type_corvidhbr:
		edid = c_edid_corvidhbr;
		size = sizeof(c_edid_corvidhbr);
		modify_port = false;
		break;
	case ntv2_edid_type_io4k:
		edid = c_edid_io4k;
		size = sizeof(c_edid_io4k);
		modify_port = false;
		break;
	case ntv2_edid_type_io4kplus:
		edid = c_edid_io4kplus;
		size = sizeof(c_edid_io4kplus);
		modify_port = false;
		break;
	case ntv2_edid_type_iox3:
		edid = c_edid_iox3;
		size = sizeof(c_edid_iox3);
		modify_port = false;
		break;
	default:
		NTV2_MSG_ERROR("%s: *error* unknown edid type %d\n", ntv2_hed->name, type);
		return NTV2_STATUS_FAIL;
	}

	if (size > NTV2_HDMIEDID_SIZE) {
		NTV2_MSG_ERROR("%s: *error* unexpected edid size %d\n", ntv2_hed->name, size);
		return NTV2_STATUS_FAIL;
	}

	if (port_index >= sizeof(c_hdmi_port_number)) {
		NTV2_MSG_ERROR("%s: *error* unexpected port index %d\n", ntv2_hed->name, port_index);
		return NTV2_STATUS_FAIL;
	}

	/* copy static edid to buffer */
	memcpy(ntv2_hed->edid_data, edid, size);
	ntv2_hed->edid_size = size;

	if (modify_port) {
		/* locate and modify edid port number */
		prefix_size = sizeof(c_hdmi_port_pefix);
		edid_size = size - prefix_size - 1;
		for (i = 0; i < edid_size; i++) {
			for (j = 0; j < prefix_size; j++) {
				if (c_hdmi_port_pefix[j] != ntv2_hed->edid_data[i + j])
					break;
			}
			if (j == prefix_size) {
				ntv2_hed->edid_data[i + prefix_size] = c_hdmi_port_number[port_index];
				break;
			}
		}
		if (i == edid_size) {
			NTV2_MSG_ERROR("%s: *error* cannot modify port number\n", ntv2_hed->name);
		}
	}

	/* compute new checksums */
	count = ntv2_hed->edid_size / 128;
	for (i = 0; i < count; i++) {
		sum = 0;
		start = i * 128;
		end = start + 127;
		for (j = start; j < end; j++) {
			sum += ntv2_hed->edid_data[j];
		}
		ntv2_hed->edid_data[end] = (uint8_t)((~sum + 1) & 0xff);
		NTV2_MSG_INFO("%s: checksum %02x\n", ntv2_hed->name, ntv2_hed->edid_data[end]);
	}

	return NTV2_STATUS_SUCCESS;
}

uint8_t *ntv2_hdmi_get_edid_data(struct ntv2_hdmiedid *ntv2_hed)
{
	if (ntv2_hed == NULL) 
		return NULL;
	
	return ntv2_hed->edid_data;
}

uint32_t ntv2_hdmi_get_edid_size(struct ntv2_hdmiedid *ntv2_hed)
{
	if (ntv2_hed == NULL) 
		return 0;
	
	return ntv2_hed->edid_size;
}

